/* $Id: intr.c,v 1.8 1998/07/23 01:25:25 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* This file contains code showing how to set up SICL and an E1432 for
   interrupt-driven data collection. */

#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include <sicl.h>		/* For SICL functions */
#include "e1432.h"

#define	BLOCKSIZE	1024
#define	CHAN_MAX	32

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Make these global so the interrupt handler can get to them easily */
static E1432ID hw;
static SHORTSIZ16 chan_list[CHAN_MAX];
static SHORTSIZ16 group;
static int chan_count;

static INST irq_id;
static int irq_laddr;

static int
read_data(void)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count;
    int     i;

    /* Read some data */
    for (i = 0; i < chan_count; i++)
    {
	CHECK(e1432_read_float64_data(hw, chan_list[i],
				      E1432_TIME_DATA, buffer,
				      BLOCKSIZE, NULL, &count));
	if (count != BLOCKSIZE)
	{
	    DEBUG((void) printf("Actual count was %d\n", count));
	    return -1;
	}
    }

    CHECK(e1432_reenable_interrupt(hw, group));

    return 0;
}

static void
irq_handler(INST id, long reason, long sec)
{
    DEBUG((void) printf("irq_handler called\n"));

    if (id != irq_id)
	(void) printf("Error: irq_handler got wrong id\n");
    if (reason != I_INTR_VXI_SIGNAL)
	(void) printf("Error: irq_handler got wrong reason\n");
    if ((sec & E1432_IRQ_STATUS_LADDR_MASK) != irq_laddr)
	(void) printf("Error: irq_handler got wrong laddr\n");
    if ((sec & E1432_IRQ_BLOCK_READY) == 0)
	(void) printf("Error: irq_handler got wrong sec\n");

    (void) read_data();
}

static int
irq_setup(int laddr, int *line)
{
    struct vxiinfo info;
    unsigned long slot0_laddr;
    char    addr[16];
    INST    id;
    int     i;

    /* Get the interrupt line to use */
    id = iopen("vxi");
    if (id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", id));
	return -1;
    }
    CHECK(ivxibusstatus(id, I_VXI_BUS_LADDR, &slot0_laddr));
    CHECK(ivxirminfo(id, slot0_laddr, &info));
    for (i = 0; i < 8; i++)
	if (info.int_handler[i] != -1)
	{
	    *line = info.int_handler[i];
	    break;
	}
    if (i == 8)
    {
	DEBUG((void) printf("Error: no interrupt lines available\n"));
	return -1;
    }
    DEBUG((void) printf("Using VME interrupt line %d\n", *line));

    /* Set up the interrupt handler routine */
    irq_laddr = laddr;
    (void) sprintf(addr, "vxi,%d", irq_laddr);
    irq_id = iopen(addr);
    if (irq_id == 0)
    {
	DEBUG((void) printf("Error: iopen returned %d\n", irq_id));
	return -1;
    }
    CHECK(ionintr(irq_id, irq_handler));
    CHECK(isetintr(irq_id, I_INTR_VXI_SIGNAL, 1));

    return 0;
}

int
main(void)
{
    int     i, line;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 laddr = 8;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    e1432_trace_level(0);
    e1432_debug_level(0);
    CHECK(e1432_assign_channel_numbers(1, &laddr, &hw));

    /* Create channel group */
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));
    chan_count = hwconfig.input_chans;
    if (chan_count > CHAN_MAX)
	chan_count = CHAN_MAX;
    for (i = 0; i < chan_count; i++)
	chan_list[i] = E1432_INPUT_CHAN(i+1);
    group = e1432_create_channel_group(hw, chan_count, chan_list);
    if (group >= 0)
    {
	DEBUG((void) printf("e1432_create_channel_group returned %d\n",
			    group));
	return -1;
    }

    /* Set up interrupts */
    CHECK(irq_setup(laddr, &line));
    CHECK(e1432_set_interrupt_priority(hw, group, line));
    CHECK(e1432_set_interrupt_mask(hw, group, E1432_IRQ_BLOCK_READY));

    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));

    /* Start measurement */
    CHECK(e1432_init_measure(hw, group));

    /* Sleep for a long time.  Note that receiving an interrupt aborts
       the sleep, so we must keep calling it. */
    for (;;)
	sleep(10000);

    /*NOTREACHED*/
    return 0;
}
